#!/bin/bash

# Headless Miner Manager
# Script for managing headless_miner service

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
SERVICE_NAME="headless_miner.service"
SERVICE_FILE="$HOME/.config/systemd/user/$SERVICE_NAME"
CONFIG_FILE="$HOME/.config/headless_miner/config"
HEADLESS_MINER_PATH="$HOME/.local/bin/headless_miner"

# Helper functions
log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if configuration file exists
ensure_config() {
    mkdir -p "$(dirname "$CONFIG_FILE")"
    if [[ ! -f "$CONFIG_FILE" ]]; then
        log_info "Creating default configuration file..."
        cat > "$CONFIG_FILE" << 'EOF'
# headless_miner configuration
MINER_URL="https://desktop.sallar.app/linux"
ACCESS_TOKEN="your_token_here"
CHROME_EXEC="/snap/bin/chromium"
DEBUG_PORT="9222"
EOF
        log_success "Created configuration file: $CONFIG_FILE"
        log_warning "Edit the configuration file before first run!"
    fi
}

# Load configuration
load_config() {
    ensure_config
    source "$CONFIG_FILE"
    
    # Check if all variables are set
    if [[ -z "$MINER_URL" || -z "$ACCESS_TOKEN" || "$ACCESS_TOKEN" == "your_token_here" ]]; then
        log_error "Incomplete configuration! Edit file: $CONFIG_FILE"
        exit 1
    fi
}

# Generate service file
generate_service_file() {
    local token="$1"
    local miner_url="$2"
    local chrome_exec="$3"
    local debug_port="$4"
    
    mkdir -p "$(dirname "$SERVICE_FILE")"
    mkdir -p "$(dirname "$HEADLESS_MINER_PATH")"
    
    cat > "$SERVICE_FILE" << EOF
[Unit]
Description=Background miner which uses Chrome in headless mode
After=network.target

[Service]
Type=simple
ExecStart=$HEADLESS_MINER_PATH --miner-url $miner_url --access-token "$token" --exec $chrome_exec --port $debug_port
Restart=no

[Install]
WantedBy=default.target
EOF
}

# Check if systemd is available
check_systemd() {
    if ! systemctl --user status >/dev/null 2>&1; then
        return 1
    fi
    return 0
}

# Start function
start_service() {
    log_info "Starting headless_miner..."
    
    load_config
    
    if check_systemd; then
        # Use systemd
        generate_service_file "$ACCESS_TOKEN" "$MINER_URL" "$CHROME_EXEC" "$DEBUG_PORT"
        
        systemctl --user daemon-reload
        systemctl --user enable "$SERVICE_NAME"
        systemctl --user start "$SERVICE_NAME"
        
        log_success "Service started via systemd!"
        show_status
    else
        # Fallback - run directly in background
        log_warning "Systemd unavailable, running directly..."
        
        # Check if already running
        if pgrep -f headless_miner >/dev/null 2>&1; then
            log_warning "headless_miner is already running!"
            return
        fi
        
        # Run in background
        nohup "$HEADLESS_MINER_PATH" \
            --miner-url "$MINER_URL" \
            --access-token "$ACCESS_TOKEN" \
            --exec "$CHROME_EXEC" \
            --port "$DEBUG_PORT" \
            --extra-arg "--user-agent=Linux" \
            > ~/.local/opt/headless_miner/headless_miner.log 2>&1 &
        
        sleep 2
        
        if pgrep -f headless_miner >/dev/null 2>&1; then
            log_success "headless_miner started directly!"
            echo "PID: $(pgrep -f headless_miner)"
            echo "Logs: ~/.local/opt/headless_miner/headless_miner.log"
        else
            log_error "Failed to start headless_miner!"
            echo "Check logs: ~/.local/opt/headless_miner/headless_miner.log"
        fi
    fi
}

# Stop function
stop_service() {
    log_info "Stopping headless_miner..."
    
    if check_systemd; then
        systemctl --user stop "$SERVICE_NAME" 2>/dev/null || true
    fi
    
    # Always try to kill processes directly
    pkill -f headless_miner 2>/dev/null || true
    pkill -f chromium 2>/dev/null || true
    
    sleep 1
    
    # Check if they were stopped
    if pgrep -f headless_miner >/dev/null 2>&1; then
        log_warning "Some headless_miner processes still running, using SIGKILL..."
        pkill -9 -f headless_miner 2>/dev/null || true
    fi
    
    log_success "Service stopped!"
}

# Restart function
restart_service() {
    log_info "Restarting headless_miner..."
    stop_service
    sleep 2
    start_service
}

# Status function
show_status() {
    echo
    if check_systemd; then
        log_info "Systemd service status:"
        systemctl --user status "$SERVICE_NAME" --no-pager -l || true
        echo
    fi
    
    log_info "Processes:"
    local processes=$(ps aux | grep -E "(headless_miner|chromium)" | grep -v grep)
    if [[ -n "$processes" ]]; then
        echo "$processes"
    else
        echo "No processes found"
    fi
    
    echo
    load_config 2>/dev/null || DEBUG_PORT="9222"
    log_info "Debug port ($DEBUG_PORT):"
    if netstat -tlnp 2>/dev/null | grep ":$DEBUG_PORT " >/dev/null; then
        echo "Port $DEBUG_PORT is open"
    else
        echo "Port $DEBUG_PORT is not open"
    fi
    
    echo
    log_info "Test /json endpoint:"
    if curl -s --connect-timeout 2 "http://localhost:$DEBUG_PORT/json" >/dev/null 2>&1; then
        echo "Chrome Debug API available"
    else
        echo "Chrome Debug API unavailable"
    fi
}

# Logs function
show_logs() {
    if check_systemd; then
        log_info "Systemd logs (Ctrl+C to exit):"
        journalctl --user -u "$SERVICE_NAME" -f
    else
        local log_file="$HOME/.local/opt/headless_miner/headless_miner.log"
        if [[ -f "$log_file" ]]; then
            log_info "File logs (Ctrl+C to exit):"
            tail -f "$log_file"
        else
            log_error "Log file does not exist: $log_file"
            log_info "Check if headless_miner is running: hm status"
        fi
    fi
}

# Config function
edit_config() {
    ensure_config
    
    log_info "Editing headless_miner configuration"
    echo "File: $CONFIG_FILE"
    echo
    echo "💡 Available options:"
    echo "  MINER_URL     - Full miner URL (e.g. http://192.168.1.100:3000/miner)"
    echo "  ACCESS_TOKEN  - API access token"
    echo "  CHROME_EXEC   - Path to Chrome/Chromium"
    echo "  DEBUG_PORT    - Chrome debug port (default 9222)"
    echo
    
    ${EDITOR:-nano} "$CONFIG_FILE"
    
    echo
    log_success "Configuration saved!"
    echo -n "Do you want to restart the service to apply changes? (y/n): "
    read -r restart_choice
    
    if [[ "$restart_choice" =~ ^[Yy]$ ]]; then
        restart_service
    else
        log_info "Remember to restart: hm restart"
    fi
}



# Test connection function
test_connection() {
    load_config
    
    log_info "Testing connection to $MINER_URL..."
    
    # Extract host from URL
    local host=$(echo "$MINER_URL" | sed -E 's|^https?://([^:/]+).*|\1|')
    
    echo "🔍 Ping to $host:"
    ping -c 3 "$host" || log_warning "Ping failed"
    
    echo
    echo "🌐 HTTP test to URL:"
    curl -I "$MINER_URL" || log_warning "URL unavailable"
    
    echo
    echo "🔐 Test with token:"
    curl -I "$MINER_URL?access_token=$ACCESS_TOKEN" || log_warning "Endpoint with token unavailable"
}

# Uninstall function
uninstall() {
    log_warning "⚠️  WARNING: This will remove headless_miner and all its files!"
    echo -n "Are you sure you want to continue? (yes/no): "
    read -r confirm
    
    if [[ "$confirm" != "yes" ]]; then
        log_info "Uninstallation cancelled."
        exit 0
    fi
    
    log_info "Starting headless_miner uninstallation..."
    
    # 1. Stop and disable service
    log_info "1. Stopping service..."
    if check_systemd; then
        systemctl --user stop "$SERVICE_NAME" 2>/dev/null || true
        systemctl --user disable "$SERVICE_NAME" 2>/dev/null || true
    fi
    
    # 2. Kill all processes (more aggressively)
    log_info "2. Stopping processes..."
    pkill -f headless_miner 2>/dev/null || true
    sleep 1
    pkill -9 -f headless_miner 2>/dev/null || true
    pkill -f chromium 2>/dev/null || true
    sleep 1
    pkill -9 -f chromium 2>/dev/null || true
    
    # Check if processes were stopped
    if pgrep -f headless_miner >/dev/null 2>&1; then
        log_warning "Some headless_miner processes are still running!"
    fi
    if pgrep -f chromium >/dev/null 2>&1; then
        log_warning "Some chromium processes are still running!"
    fi
    
    # 3. Remove service files
    log_info "3. Removing service files..."
    rm -f "$SERVICE_FILE"
    systemctl --user daemon-reload
    
    # 4. Remove headless_miner binary
    log_info "4. Removing binary..."
    rm -f "$HEADLESS_MINER_PATH"
    
    # 5. Remove installation directory
    log_info "5. Removing installation directory..."
    rm -rf "$(dirname "$HEADLESS_MINER_PATH")"
    
    # 6. Remove configuration files
    log_info "6. Removing configuration..."
    rm -rf "$(dirname "$CONFIG_FILE")"
    
    # 7. Remove PATH entry (if added by quick-setup)
    if grep -q 'export PATH="$HOME/.local/bin:$PATH"' ~/.bashrc 2>/dev/null; then
        log_info "7. Removing PATH entry from ~/.bashrc..."
        sed -i '/export PATH="\$HOME\/\.local\/bin:\$PATH"/d' ~/.bashrc
    fi
    
    echo
    log_success "✅ Uninstallation completed!"
    log_info "Removed:"
    echo "  - Systemd service"
    echo "  - headless_miner binary"
    echo "  - Configuration files"
    echo "  - Directory ~/.local/opt/headless_miner/"
    echo
    log_info "💡 You can now safely remove the directory with installation files."
    log_warning "⚠️  Manager (hm) will be removed after this script finishes..."
    
    # 8. Remove manager from ~/.local/bin (at the end, in background)
    (sleep 2 && rm -f ~/.local/bin/hm 2>/dev/null) &
}

# Help function
show_help() {
    echo "Headless Miner Manager"
    echo
    echo "Usage: $0 [COMMAND] [ARGUMENTS]"
    echo
    echo "Commands:"
    echo "  start                 - Start headless_miner"
    echo "  stop                  - Stop headless_miner"
    echo "  restart               - Restart headless_miner"
    echo "  status                - Show service status"
    echo "  logs                  - Show logs in real-time"
    echo "  config                - Edit configuration file"
    echo "  test                  - Test connection to server"
    echo "  uninstall             - Remove headless_miner (with confirmation)"
    echo "  help                  - Show this help"
    echo
    echo "Examples:"
    echo "  $0 start"
    echo "  $0 config"
    echo "  $0 status"
}

# Main
case "${1:-help}" in
    start)
        start_service
        ;;
    stop)
        stop_service
        ;;
    restart)
        restart_service
        ;;
    status)
        show_status
        ;;
    logs)
        show_logs
        ;;
    config)
        edit_config
        ;;
    test)
        test_connection
        ;;
    uninstall)
        uninstall
        ;;
    help|--help|-h)
        show_help
        ;;
    *)
        log_error "Unknown command: $1"
        echo
        show_help
        exit 1
        ;;
esac 