#!/bin/bash

# Installation script for headless_miner on ARM64 Linux

set -e

echo "🚀 Installing headless_miner (ARM64)..."

# Check architecture
ARCH=$(uname -m)
if [[ "$ARCH" != "aarch64" && "$ARCH" != "arm64" ]]; then
    echo "⚠️  Warning: This version is built for ARM64, but detected architecture: $ARCH"
    echo "   Continue? (y/N)"
    read -r response
    if [[ ! "$response" =~ ^[Yy]$ ]]; then
        echo "❌ Installation cancelled"
        exit 1
    fi
fi

# Check if running as root for system installation
if [[ $EUID -eq 0 ]]; then
    INSTALL_DIR="/usr/local/bin"
    SERVICE_DIR="/etc/systemd/user"
    echo "📦 System installation in $INSTALL_DIR"
else
    INSTALL_DIR="$HOME/.local/bin"
    SERVICE_DIR="$HOME/.config/systemd/user"
    echo "👤 User installation in $INSTALL_DIR"
    
    # Create directories if they don't exist
    mkdir -p "$INSTALL_DIR"
    mkdir -p "$SERVICE_DIR"
fi

# Copy executable file
echo "📋 Copying executable file..."
cp headless_miner "$INSTALL_DIR/"
chmod +x "$INSTALL_DIR/headless_miner"

# Copy service file
echo "⚙️  Copying systemd service file..."
cp headless_miner.service "$SERVICE_DIR/"

# Add to PATH if user installation
if [[ $EUID -ne 0 ]]; then
    if ! echo "$PATH" | grep -q "$HOME/.local/bin"; then
        echo "🔧 Adding $HOME/.local/bin to PATH..."
        echo 'export PATH="$HOME/.local/bin:$PATH"' >> ~/.bashrc
        echo "⚠️  Run 'source ~/.bashrc' or open a new terminal"
    fi
fi

# Check if Chrome is installed
echo "🔍 Checking for Chrome..."
if command -v google-chrome >/dev/null 2>&1; then
    echo "✅ Google Chrome found: $(google-chrome --version)"
elif command -v chromium-browser >/dev/null 2>&1; then
    echo "✅ Chromium found: $(chromium-browser --version)"
elif command -v chromium >/dev/null 2>&1; then
    echo "✅ Chromium found: $(chromium --version)"
else
    echo "⚠️  Chrome/Chromium not found!"
    echo "   Install Chrome or Chromium:"
    echo "   Ubuntu/Debian: sudo apt install chromium-browser"
    echo "   CentOS/RHEL: sudo dnf install chromium"
    echo "   Arch: sudo pacman -S chromium"
fi

# Copy management scripts (if they exist)
if [[ -f "headless-miner-manager.sh" ]]; then
    echo "📋 Installing management scripts..."
    cp headless-miner-manager.sh "$INSTALL_DIR/hm"
    chmod +x "$INSTALL_DIR/hm"
    echo "✅ Installed manager as 'hm'"
fi

if [[ -f "quick-setup.sh" ]]; then
    echo "🔧 Running quick setup..."
    chmod +x quick-setup.sh
    ./quick-setup.sh
else
    echo "✅ Installation completed!"
    echo ""
    echo "🔧 Usage:"
    echo "   headless_miner --help"
    echo "   headless_miner --miner-url http://localhost:3000/miner --access-token YOUR_TOKEN"
    echo ""
    echo "🔄 Service management:"
    if [[ -f "$INSTALL_DIR/hm" ]]; then
        echo "   hm start              - Start service"
        echo "   hm stop               - Stop service"
        echo "   hm restart            - Restart service"
        echo "   hm status             - Service status"
        echo "   hm config             - Edit configuration"
        echo "   hm test               - Test connection"
        echo "   hm logs               - View logs"
        echo "   hm uninstall          - Remove headless_miner"
    else
        echo "   systemctl --user start headless_miner.service"
        echo "   systemctl --user enable headless_miner.service"
        echo "   systemctl --user status headless_miner.service"
    fi
    echo ""
    echo "🗑️  Uninstallation:"
    echo "   hm uninstall          - Remove via manager"
    echo "   ./uninstall.sh        - Remove via script"
    echo ""
    echo "🏗️  Architecture: ARM64 (aarch64)"
fi 