#!/bin/bash

# Standalone Uninstall Script for headless_miner
# Can be run independently from the manager

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Logging functions
log_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

log_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

log_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

log_error() {
    echo -e "${RED}❌ $1${NC}"
}

# Paths
SERVICE_NAME="headless_miner.service"
SERVICE_FILE="$HOME/.config/systemd/user/$SERVICE_NAME"
HEADLESS_MINER_PATH="$HOME/.local/opt/headless_miner/headless_miner"
CONFIG_FILE="$HOME/.config/headless_miner/config"

echo "🗑️  headless_miner Uninstaller"
echo "==============================="
echo

log_warning "⚠️  WARNING: This will remove headless_miner and all its files!"
echo "The following will be removed:"
echo "  - Systemd service ($SERVICE_FILE)"
echo "  - headless_miner binary ($HEADLESS_MINER_PATH)"
echo "  - Configuration files ($CONFIG_FILE)"
echo "  - Manager hm (~/.local/bin/hm)"
echo "  - Installation directory (~/.local/opt/headless_miner/)"
echo "  - PATH entry from ~/.bashrc (if added)"
echo

echo -n "Are you sure you want to continue? (yes/no): "
read -r confirm

if [[ "$confirm" != "yes" ]]; then
    log_info "Uninstallation cancelled."
    exit 0
fi

echo
log_info "Starting headless_miner uninstallation..."

# 1. Stop and disable service
log_info "1. Stopping service..."
systemctl --user stop "$SERVICE_NAME" 2>/dev/null || true
systemctl --user disable "$SERVICE_NAME" 2>/dev/null || true

# 2. Kill all processes (more aggressively)
log_info "2. Stopping processes..."
pkill -f headless_miner 2>/dev/null || true
sleep 1
pkill -9 -f headless_miner 2>/dev/null || true
pkill -f chromium 2>/dev/null || true
sleep 1
pkill -9 -f chromium 2>/dev/null || true

# Check if processes were stopped
if pgrep -f headless_miner >/dev/null 2>&1; then
    log_warning "Some headless_miner processes are still running!"
fi
if pgrep -f chromium >/dev/null 2>&1; then
    log_warning "Some chromium processes are still running!"
fi

# 3. Remove service files
log_info "3. Removing service files..."
rm -f "$SERVICE_FILE"
systemctl --user daemon-reload

# 4. Remove headless_miner binary
log_info "4. Removing binary..."
rm -f "$HEADLESS_MINER_PATH"

# 5. Remove installation directory
log_info "5. Removing installation directory..."
rm -rf "$(dirname "$HEADLESS_MINER_PATH")"

# 6. Remove configuration files
log_info "6. Removing configuration..."
rm -rf "$(dirname "$CONFIG_FILE")"

# 7. Remove manager from ~/.local/bin
log_info "7. Removing manager..."
rm -f ~/.local/bin/hm

# 8. Remove PATH entry (if added by quick-setup)
if grep -q 'export PATH="$HOME/.local/bin:$PATH"' ~/.bashrc 2>/dev/null; then
    log_info "8. Removing PATH entry from ~/.bashrc..."
    sed -i '/export PATH="\$HOME\/\.local\/bin:\$PATH"/d' ~/.bashrc
    log_warning "Run 'source ~/.bashrc' or open a new terminal to update PATH"
fi

echo
log_success "✅ Uninstallation completed!"
echo
log_info "Removed:"
echo "  ✓ Systemd service"
echo "  ✓ headless_miner binary"
echo "  ✓ Configuration files"
echo "  ✓ Manager (hm)"
echo "  ✓ Directory ~/.local/opt/headless_miner/"
echo

log_info "💡 You can now safely remove this directory with installation files."
echo
log_info "Thank you for using headless_miner! 👋" 